<?php

namespace App\Livewire;

use App\Models\SmsLog;
use App\Models\Loan;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Livewire\Component;

class RepaymentAlertsView extends Component
{
    public $paymentPlan = '';
    public $filterDate;
    public $reminderType = '';

    // Search bars
    public $searchUpcoming = '';
    public $searchMissed = '';
    public $searchLogs = '';

    public function mount()
    {
        $this->filterDate = Carbon::now()->format('Y-m-d');
    }

    // Upcoming repayments (Next 7 days)
    public function getUpcomingRepaymentsProperty()
    {
        $query = Loan::with(['user', 'payments', 'reminderLogs', 'loanPackage'])
            ->when($this->paymentPlan, fn($q) => $q->where('loan_payment_plan', $this->paymentPlan));

        if ($this->searchUpcoming) {
            $search = strtolower($this->searchUpcoming);
            $query->whereHas('user', function ($q) use ($search) {
                $q->whereRaw('LOWER(formal_name) LIKE ?', ["%{$search}%"]);
            });
        }

        return $query->get()->filter(function ($loan) {
            $dueDate = $loan->due_date;
            if (!$dueDate) {
                Log::warning('Loan missing due_date', ['loan_id' => $loan->id]);
                return false;
            }

            $isDueSoon = $dueDate->isBetween(Carbon::today(), Carbon::today()->addDays(7));
            $hasOutstanding = ($loan->loan_required_amount - $loan->payments->sum('paid_amount')) > 0;

            if (!$loan->loanPackage) {
                Log::warning('Loan missing loanPackage', ['loan_id' => $loan->id]);
            }

            return $isDueSoon && $hasOutstanding;
        })->map(function ($loan) {
            $loan->amount_due = $loan->loan_required_amount - $loan->payments->sum('paid_amount');
            $loan->reminders = $loan->reminderLogs->groupBy('type')->map->first();
            return $loan;
        });
    }

    // Missed repayments (Overdue)
    public function getMissedRepaymentsProperty()
    {
        $query = Loan::with(['user', 'payments', 'reminderLogs', 'loanPackage'])
            ->when($this->paymentPlan, fn($q) => $q->where('loan_payment_plan', $this->paymentPlan));

        if ($this->searchMissed) {
            $search = strtolower($this->searchMissed);
            $query->whereHas('user', function ($q) use ($search) {
                $q->whereRaw('LOWER(formal_name) LIKE ?', ["%{$search}%"]);
            });
        }

        return $query->get()->filter(function ($loan) {
            $dueDate = $loan->due_date;
            $hasOutstanding = ($loan->loan_required_amount - $loan->payments->sum('paid_amount')) > 0;

            return $dueDate && $dueDate->lt(Carbon::today()) && $hasOutstanding;
        })->map(function ($loan) {
            $loan->amount_due = $loan->loan_required_amount - $loan->payments->sum('paid_amount');
            $loan->days_overdue = $loan->due_date->diffInDays(Carbon::today());
            $loan->reminder_after = $loan->reminderLogs->firstWhere('type', 'after');
            return $loan;
        });
    }

    // Reminder Logs
    public function getReminderLogsProperty()
    {
        $query = SmsLog::query()->whereDate('sent_at', $this->filterDate);

        if ($this->searchLogs) {
            $search = strtolower($this->searchLogs);
            $query->where(function ($q) use ($search) {
                $q->whereRaw('LOWER(applicant_name) LIKE ?', ["%{$search}%"])
                  ->orWhereRaw('CAST(loan_id AS CHAR) LIKE ?', ["%{$search}%"]);
            });
        }

        return $query->orderByDesc('sent_at')->take(100)->get();
    }

    public function render()
    {
        return view('livewire.repayment-alerts-view', [
            'upcomingRepayments' => $this->upcomingRepayments,
            'missedRepayments' => $this->missedRepayments,
            'sms_logs' => $this->reminderLogs,
        ]);
    }
}
